--[[
Copyright (c) 2017 Vexlio, LLC. All rights reserved. 

This file is licensed under the MIT License:

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

-- Plugin info
VexlioPlugin = {
    name = "Connect common tangents",
    author = "Vexlio",
    description = "Connects two selected circles with their common tangent lines.",
    id = "com.vexlio.common-tangents"
}

-- Handles for UI components
local drawButton

-- Return a circle "struct"
local function make_circle(center, radius)
    return {center=center, radius=radius}
end

-- Return true if the two floating point values are close enough to be
-- considered equal.
local function eq(float1, float2)
    local eps = 1e-5
    return math.abs(float1 - float2) <= eps
end

-- Return the point p1 or p2 that is closest to point q.
local function closestTo(p1, p2, q)
    local d1, d2 = Vexlio:dist(p1, q), Vexlio:dist(p2, q)
    if d1 <= d2 then
        return p1
    else
        return p2
    end
end

-- Return the point p1 or p2 that is farthest from point q.
local function farthestFrom(p1, p2, q)
    local d1, d2 = Vexlio:dist(p1, q), Vexlio:dist(p2, q)
    if d1 <= d2 then
        return p2
    else
        return p1
    end
end

-- Return true if one circle completely encloses the other.
local function circleContains(c1, c2)
    local d = Vexlio:dist(c1.center, c2.center)
    return d + math.min(c1.radius, c2.radius) <= math.max(c1.radius, c2.radius)
end

-- Return the first two circles in the selection.
local function firstTwoSelectedCircles()
    local n = Vexlio.drawing.numSelected
    local c1, c2 = nil, nil
    for i = 0, n-1 do
        local obj = Vexlio.drawing:getSelectedObject(i)
        if obj.type == Type.Ellipse and eq(obj.rx, obj.ry) then
            local c = make_circle(pt(obj.cx, obj.cy), obj.rx)
            if c1 == nil then
                c1 = c
            elseif c2 == nil then
                c2 = c
            else
                break
            end
        end
    end
    return c1, c2
end

-- Return 2 pairs of points that, when connected, form the common
-- tangents of the given circles. For an explanation and proof of the
-- method used here, see e.g.:
-- http://www.mathopenref.com/consttangentsext.html
local function common_tangents(c1, c2)
    local A, B = c1.center, c2.center
    local r1, r2 = c1.radius, c2.radius
    local F1, L1, F2, L2

    -- Degenerate case: no tangent lines.
    if circleContains(c1, c2) then
        return { {nil, nil}, {nil, nil} }
    end

    -- Ensure B (r2) is larger.
    if r1 > r2 then
        A, B = B, A
        r1, r2 = r2, r1
    end

    -- For the degenerate case where r1 == r2, any tangent is a common tangent.
    -- Just pick the two with the same angle as between the two center points.
    if eq(r1, r2) then
        _, F1, F2 = Vexlio:circleLineIntersect(A, r1, A, Vexlio:rotatePoint(B, A, 90))
        _, L1, L2 = Vexlio:circleLineIntersect(B, r2, B, Vexlio:rotatePoint(A, B, 90))
    else
        local dr = r2 - r1
        local M = Vexlio:midpoint(A, B)
        local cA = make_circle(B, dr)
        local cM = make_circle(M, Vexlio:dist(M, B))
        local _, J1, J2 = Vexlio:circleCircleIntersect(cM.center, cM.radius, cA.center, cA.radius)
        local _, L1a, L1b = Vexlio:circleLineIntersect(B, r2, B, J1)
        local _, L2a, L2b = Vexlio:circleLineIntersect(B, r2, B, J2)
        L2, L1 = closestTo(L1a, L1b, A), closestTo(L2a, L2b, A)

        local u = Vexlio:normalize(pt(A.x - B.x, A.y - B.y))
        u.x = u.x * dr
        u.y = u.y * dr
        local S = pt(B.x + u.x, B.y + u.y)
        local Z = pt(A.x + u.x, A.y + u.y)
        local sj = Vexlio:dist(S, J1)
        local _, T1, T2 = Vexlio:circleCircleIntersect(A, dr, Z, sj)
        local _, F1a, F1b = Vexlio:circleLineIntersect(A, r1, A, T1)
        local _, F2a, F2b = Vexlio:circleLineIntersect(A, r1, A, T2)
        F1, F2 = farthestFrom(F1a, F1b, B), farthestFrom(F2a, F2b, B)
    end

    -- Tangent lines are from F1->L1 and F2->L2
    return { {F1, L1}, {F2, L2} }
end

-- Callback from the draw tangents button.
local function drawTangents()
    local c1, c2 = firstTwoSelectedCircles()
    -- Compute the tangents and draw them.
    if c1 ~= nil and c2 ~= nil then
        local result = common_tangents(c1, c2)
        local C, D = result[1][1], result[1][2]
        local E, F = result[2][1], result[2][2]
        if C ~= nil and D ~= nil then
            line(C, D)
        end
        if E ~= nil and F ~= nil then
            line(E, F)
        end
        return true
    end
    return false
end

-- Called by Vexlio: Implement this function to receive a callback
-- whenever the selection in the drawing changes.
function onSelectionChange()
    local c1, c2 = firstTwoSelectedCircles()
    drawButton.enabled = c1 ~= nil and c2 ~= nil
end

-- Called by Vexlio: Implement this function to control whether the
-- plugin displays a UI (window) or not.
function pluginShouldShowUI()
    -- Try to draw tangent lines on the current selection. We only want
    -- to display the UI if that fails.
    return not drawTangents()
end

-- Called by Vexlio: Implement this function to set up any UI needed
-- by the plugin.
function pluginUI()
    local win = Vexlio:window()
    win.title = "Common tangents"
    win:label("Select exactly two circles in the drawing.", win:row(), 0, 12)
    win:row(10) -- Spacer
    drawButton = win:button("Draw Tangents", drawTangents, win:row(), 3, 6)
    drawButton.enabled = false
end