--[[
Copyright (c) 2017 Vexlio, LLC. All rights reserved. 

This file is licensed under the MIT License:

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

VexlioPlugin = {
    name = "Create spirals",
    author = "Vexlio",
    description = "Creates Archimedean spirals in the drawing.",
    id = "com.vexlio.create-spirals"
}

-- Handles for UI components
local spacingSlider, nturnsSlider

-- Other globals
local spiral = nil
local spiralCenter = nil

-- Return the derivative of the spiral at theta parameterized by a and
-- b. Recall the polar form equation of the Archimedean spiral is r =
-- a + b*theta.
function spiralDeriv(a, b, theta)
    local c, s = math.cos(theta), math.sin(theta)
    return (b*s + (a + b*theta)*c)/(b*c - (a+b*theta)*s)
end

-- Creates a new spiral with the given center point.
function drawSpiral(center)
    local a = 0
    local b = spacingSlider.value
    local nturns = nturnsSlider.value
    local dtheta = math.rad(45) -- Sample distance for Bezier segment creation
    local niters = nturns * 2 * math.pi / dtheta
    
    spiral = nil

    for i = 0, niters - 1 do
        local ta = dtheta*i
        local tb = dtheta*(i+1)
        -- Compute the spiral r values at the two theta values
        local r1, r2 = a + b * ta, a + b * tb
        -- Compute the points on the spiral at the two theta values.
        -- These will be the two endpoints of the Bezier segment.
        local p1 = pt(r1 * math.cos(ta), r1 * math.sin(ta))
        local p2 = pt(r2 * math.cos(tb), r2 * math.sin(tb))
        -- Derivative of spiral at each of the two points.
        local d1, d2 = spiralDeriv(a, b, ta), spiralDeriv(a, b, tb)
        -- Compute the intersection of the lines tangent to the spiral
        -- at each of the two points. The result is the quadratic
        -- Bezier control point.
        local _, I = Vexlio:lineIntersect(p1, pt(p1.x + 10, d1*10 + p1.y), 
                                          p2, pt(p2.x + 10, d2*10 + p2.y))

        -- Move from origin 0,0 to the desired center point.
        p1.x = p1.x + center.x
        p1.y = p1.y + center.y
        p2.x = p2.x + center.x
        p2.y = p2.y + center.y
        I.x = I.x + center.x
        I.y = I.y + center.y

        if spiral == nil then
            spiral = Vexlio.drawing.currentLayer:path(p1)
        end
        spiral:bezTo(I.x, I.y, p2.x, p2.y)   
    end
end

-- Removes and redraws the current spiral, which effectively modifies
-- it in place.
function redrawSpiral()
    if spiral ~= nil then
        spiral:delete()
        drawSpiral(spiralCenter)
    end
end

-- Handle mouse clicks to draw the spiral at the click location.
function onMouseUp(p)
    drawSpiral(p)
    spiralCenter = p
end

-- Set up UI for the plugin.
function pluginUI()
    local minSpacing, maxSpacing = 1, 50
    local minNumTurns, maxNumTurns = 1, 50

    local win = Vexlio:window()
    win:label("Click to create a spiral, then use the sliders to adjust its parameters.", win:row(), 0, 12)
    win:row(10) -- Spacer
    local r = win:row()
    win:label("Spacing: ", r, 0, 3)
    spacingSlider = win:slider(minSpacing, maxSpacing, 1, redrawSpiral, r, 3, 9)
    spacingSlider.value = 10

    win:row(10) -- Spacer
    local r = win:row()
    win:label("# turns: ", r, 0, 3)
    nturnsSlider = win:slider(1, 50, 1, redrawSpiral, r, 3, 9)
    nturnsSlider.value = 3
end
