--[[
Copyright (c) 2017 Vexlio, LLC. All rights reserved. 

This file is licensed under the MIT License:

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

-- Plugin info
VexlioPlugin = {
    name = "Cut path",
    author = "Vexlio",
    description = "Given two paths, one on top of the other, inserts path nodes in the bottom path at the intersection points, and deletes the top path.",
    id = "com.vexlio.cut-path"
}

-- Return the first two selected paths.
function getTwoSelectedPaths()
    local n = Vexlio.drawing.numSelected
    local path1, path2 = nil, nil
    for i = 0, n-1 do
        local obj = Vexlio.drawing:getSelectedObject(i)
        if obj.type == Type.Path then
            if path1 == nil then
                path1 = obj
            elseif path2 == nil then
                path2 = obj
            else
                break
            end
        end
    end
    return path1, path2
end

-- Try to perform the cut path operation, returning true if successful.
function cutPath()
    local path1, path2 = getTwoSelectedPaths()
    -- Compute the intersections.
    if path1 ~= nil and path2 ~= nil then
        -- Figure out which one is on top.
        local top, bottom = nil, nil
        if path1:isOnTopOf(path2) then
            top, bottom = path1, path2
        else
            top, bottom = path2, path1
        end
        local intersections = path1:intersectionPoints(path2)
        for p in each(intersections) do
            bottom:splitPathAtPoint(p)
        end
        top:delete()
        return true
    end
    return false
end

-- Called by Vexlio: Implement this function to control whether the
-- plugin displays a UI (window) or not.
function pluginShouldShowUI()
    -- Try to insert nodes on the current selection. We only want
    -- to display the UI if that fails.
    return not cutPath()
end

-- Called by Vexlio: Implement this function to set up any UI needed
-- by the plugin.
function pluginUI()
    local win = Vexlio:window()
    win.title = VexlioPlugin.name
    win:label("Select exactly two paths in the drawing.", win:row(), 0, 12)
    win:row(10) -- Spacer
    win:button("Cut", cutPath, win:row(), 4, 4)
end