--[[
Copyright (c) 2017 Vexlio, LLC. All rights reserved. 

This file is licensed under the MIT License:

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

-- Plugin info
VexlioPlugin = {
    name = "Find and replace styles",
    author = "Vexlio",
    description = "Find and replace object style values in a drawing.",
    id = "com.vexlio.find-replace-styles"
}

-- Handles for UI components
local find_lineWidthCheckbox, find_lineWidthTextbox
local find_lineColorCheckbox, find_lineColorSwatch
local find_fillColorCheckbox, find_fillColorSwatch
local replace_lineWidthLabel, replace_lineWidthTextbox
local replace_lineColorLabel, replace_lineColorSwatch
local replace_fillColorLabel, replace_fillColorSwatch
local findNextButton, findPrevButton, replaceButton, replaceAllButton

-- Other globals
local currentMatch

-- Based on the currently selected find options, return true if the
-- given object is a match.
function match(obj)
    local someOption = false
    local allMatching = true
    if find_lineWidthCheckbox.checked then
        allMatching = allMatching and obj.strokeWidth == find_lineWidthTextbox.floatValue
        someOption = true
    end
    if find_lineColorCheckbox.checked then
        allMatching = allMatching and obj.stroke.argb == find_lineColorSwatch.argb
        someOption = true
    end
    if find_fillColorCheckbox.checked then
        allMatching = allMatching and obj.fill.argb == find_fillColorSwatch.argb
        someOption = true
    end
    return someOption and allMatching
end

-- Based on the currently selected find options, replace all
-- properties on the given object with the new values as specified by
-- the user.
function replace(obj)
    if find_lineWidthCheckbox.checked then
        obj.strokeWidth = replace_lineWidthTextbox.floatValue
    end
    if find_lineColorCheckbox.checked then
        obj.stroke.argb = replace_lineColorSwatch.argb
    end
    if find_fillColorCheckbox.checked then
        obj.fill.argb = replace_fillColorSwatch.argb
    end
end

-- Find the first object in the drawing matching the current parameters.
function findFirstMatch()
    local obj = Vexlio.drawing:firstObject()
    while obj ~= nil and not match(obj) do
        obj = obj:successor()
    end
    if obj == nil or not match(obj) then
        return nil
    else
        return obj
    end
end

-- Find the last object in the drawing matching the current parameters.
function findLastMatch()
    local obj = Vexlio.drawing:lastObject()
    while obj ~= nil and not match(obj) do
        obj = obj:predecessor()
    end
    if obj == nil or not match(obj) then
        return nil
    else
        return obj
    end
end

-- Advance the current match to the next one. If there is no next match, return
-- nil.
function nextMatch()
    if currentMatch == nil then
        currentMatch = findFirstMatch()
    else
        currentMatch = currentMatch:successor()
        while currentMatch ~= nil and not match(currentMatch) do
            currentMatch = currentMatch:successor()
        end
    end
    if currentMatch ~= nil then
        currentMatch:select()
        Vexlio.drawing:centerViewOnObject(currentMatch)
    end
    replaceButton.enabled = currentMatch ~= nil
    replaceAllButton.enabled = currentMatch ~= nil
end

-- Move the current match to the previous one. If there is no previous match,
-- return nil.
function previousMatch()
    if currentMatch == nil then
        currentMatch = findLastMatch()
    else
        currentMatch = currentMatch:predecessor()
        while currentMatch ~= nil and not match(currentMatch) do
            currentMatch = currentMatch:predecessor()
        end
    end
    if currentMatch ~= nil then
        currentMatch:select()
        Vexlio.drawing:centerViewOnObject(currentMatch)
    end
    replaceButton.enabled = currentMatch ~= nil
    replaceAllButton.enabled = currentMatch ~= nil
end

-- Click handler for the find previous button.
function findPrevButtonClick()
    previousMatch()
end

-- Click handler for the find next button.
function findNextButtonClick()
    nextMatch()
end

-- Click handler for the replace button.
function replaceButtonClick()
    if currentMatch ~= nil then
        replace(currentMatch)
        nextMatch()
    end
end

-- Click handler for the replace all button.
function replaceAllButtonClick()
    currentMatch = findFirstMatch()
    while currentMatch ~= nil do
        replace(currentMatch)
        nextMatch()
    end
end

-- Called by Vexlio: Implement this function to set up any UI needed by the
-- plugin.
function pluginUI()
    local win = Vexlio:window()
    win.title = VexlioPlugin.name
    win:label("Find:", win:row(), 0, 12)
    win:row(5) -- Spacer

    ----- Find controls -----
    local r = win:row()
    find_lineWidthCheckbox = win:checkbox(
        "Line Width", 
        function()
            replace_lineWidthTextbox.enabled = find_lineWidthCheckbox.checked
        end, r, 1, 11)
    find_lineWidthTextbox = win:textbox(r, 8, 3)

    r = win:row()
    find_lineColorCheckbox = win:checkbox(
        "Line Color", 
        function()
            replace_lineColorSwatch.enabled = find_lineColorCheckbox.checked
        end, r, 1, 11)
    find_lineColorSwatch = win:swatch(r, 8, 3)

    r = win:row()
    find_fillColorCheckbox = win:checkbox(
        "Fill Color", 
        function()
            replace_fillColorSwatch.enabled = find_fillColorCheckbox.checked
        end, r, 1, 11)
    find_fillColorSwatch = win:swatch(r, 8, 3)

    ----- Replace controls -----
    win:row(10) -- Spacer
    win:label("Replace:", win:row(), 0, 12)
    win:row(5) -- Spacer

    local r = win:row()
    replace_lineWidthLabel = win:label("Line Width", r, 1, 11)
    replace_lineWidthTextbox = win:textbox(r, 8, 3)

    r = win:row()
    replace_lineColorLabel = win:label("Line Color", r, 1, 11)
    replace_lineColorSwatch = win:swatch(r, 8, 3)

    r = win:row()
    replace_fillColorLabel = win:label("Fill Color", r, 1, 11)
    replace_fillColorSwatch = win:swatch(r, 8, 3)

    ----- Bottom buttons -----
    win:row(10) -- Spacer
    r = win:row()
    findPrevButton = win:button("<", findPrevButtonClick, r, 0, 3)
    findNextButton = win:button(">", findNextButtonClick, r, 3, 3)
    replaceButton = win:button("Replace", replaceButtonClick, r, 6, 3)
    replaceAllButton = win:button("Replace all", replaceAllButtonClick, r, 9, 3)

    -- Set initial enabled states.
    replace_lineWidthTextbox.enabled = false
    replace_lineColorSwatch.enabled = false
    replace_fillColorSwatch.enabled = false
    findNextButton.enabled = true
    findPrevButton.enabled = true
    replaceButton.enabled = false
    replaceAllButton.enabled = false
end