--[[
Copyright (c) 2017 Vexlio, LLC. All rights reserved. 

This file is licensed under the MIT License:

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
--]]

VexlioPlugin = {
    name = "Rope toy",
    author = "Vexlio",
    description = "Creates an interactive rope that follows the mouse.",
    id = "com.vexlio.rope-toy"
}

-- Number of nodes on the rope path.
local points = 25
-- Length of a rope segment.
local length = 35
-- The rope path.
local rope = nil

-- Create a new rope starting at the given point.
local function newRope(start)
    rope = Vexlio.drawing.currentLayer:path(start.x, start.y)
    -- Setting interactive to false allows for higher performance
    -- while the mouse is moving. This is because Vexlio does not
    -- compute bounding boxes for non-interactive objects (bounding
    -- box computation can be expensive). Normally, Vexlio would
    -- recalculate the bounding box for the path every time it is
    -- modified, which in this example is on every mouse move event.
    rope.interactive = false
    rope.strokeWidth = 20
    for i = 0, points-1 do
        local p = pt(start.x + i*length, start.y)
        rope:lineTo(p.x, p.y)
    end
end

-- Handle mouse move events.
function onMouseMove(p)
    if rope == nil then
        return
    end

    -- This method is based on: http://paperjs.org/examples/chain/
    -- which in turn is based on:
    -- http://processing.org/learning/topics/follow3.html.
    rope:getNode(0).point = p
    for i = 0, points - 1 do
        local node = rope:getNode(i)
        local nextNode = rope:getNode(i+1)
        local v = pt(node.point.x - nextNode.point.x, node.point.y - nextNode.point.y)
        v = Vexlio:normalize(v)
        v.x = v.x * length
        v.y = v.y * length
        nextNode.point = pt(node.point.x - v.x, node.point.y - v.y)
    end
end

-- Handle mouse up events.
function onMouseUp(p)
   if rope == nil then
       newRope(p)
   else
       rope = nil
   end
end

-- Called by Vexlio: create the UI.
function pluginUI()
    local win = Vexlio:window()
    win:label("Click to create a rope, then move the mouse.", win:row(), 1, 10)
end